<?php
/**
 * Products API for Toko Baju Junata
 * Handles product catalog, search, filter, and details
 */

require_once 'config.php';

$requestData = getRequestData();
$action = isset($requestData['action']) ? $requestData['action'] : '';

switch ($action) {
    case 'getAll':
        getAllProducts($pdo);
        break;
    
    case 'search':
        searchProducts($pdo, $requestData);
        break;
    
    case 'filter':
        filterProducts($pdo, $requestData);
        break;
    
    case 'detail':
        getProductDetail($pdo, $requestData);
        break;
    
    case 'categories':
        getCategories($pdo);
        break;
    
    default:
        sendResponse(false, 'Invalid action', null, 400);
        break;
}

/**
 * Get all products from catalog
 */
function getAllProducts($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM v_product_catalog");
        $products = $stmt->fetchAll();
        
        sendResponse(true, 'Products retrieved successfully', $products, 200);
        
    } catch (PDOException $e) {
        logError("Get all products error: " . $e->getMessage());
        sendResponse(false, 'Failed to retrieve products', null, 500);
    }
}

/**
 * Search products by keyword
 */
function searchProducts($pdo, $data) {
    if (!isset($data['keyword']) || empty(trim($data['keyword']))) {
        sendResponse(false, 'Search keyword is required', null, 400);
    }
    
    $keyword = '%' . sanitizeInput($data['keyword']) . '%';
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM v_product_catalog 
            WHERE nama LIKE ? OR deskripsi LIKE ?
            ORDER BY created_at DESC
        ");
        $stmt->execute([$keyword, $keyword]);
        $products = $stmt->fetchAll();
        
        sendResponse(true, 'Search completed', $products, 200);
        
    } catch (PDOException $e) {
        logError("Search products error: " . $e->getMessage());
        sendResponse(false, 'Search failed', null, 500);
    }
}

/**
 * Filter products by category
 */
function filterProducts($pdo, $data) {
    if (!isset($data['kategori']) || empty(trim($data['kategori']))) {
        sendResponse(false, 'Category is required', null, 400);
    }
    
    $kategori = sanitizeInput($data['kategori']);
    
    try {
        $stmt = $pdo->prepare("
            SELECT * FROM v_product_catalog 
            WHERE kategori = ?
            ORDER BY created_at DESC
        ");
        $stmt->execute([$kategori]);
        $products = $stmt->fetchAll();
        
        sendResponse(true, 'Products filtered successfully', $products, 200);
        
    } catch (PDOException $e) {
        logError("Filter products error: " . $e->getMessage());
        sendResponse(false, 'Filter failed', null, 500);
    }
}

/**
 * Get product detail by ID
 */
function getProductDetail($pdo, $data) {
    if (!isset($data['id']) || empty($data['id'])) {
        sendResponse(false, 'Product ID is required', null, 400);
    }
    
    $id = intval($data['id']);
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM v_product_catalog WHERE id = ?");
        $stmt->execute([$id]);
        $product = $stmt->fetch();
        
        if (!$product) {
            sendResponse(false, 'Product not found', null, 404);
        }
        
        sendResponse(true, 'Product detail retrieved', $product, 200);
        
    } catch (PDOException $e) {
        logError("Get product detail error: " . $e->getMessage());
        sendResponse(false, 'Failed to retrieve product detail', null, 500);
    }
}

/**
 * Get all product categories
 */
function getCategories($pdo) {
    try {
        $stmt = $pdo->query("
            SELECT DISTINCT kategori, COUNT(*) as total_products 
            FROM products 
            GROUP BY kategori 
            ORDER BY kategori
        ");
        $categories = $stmt->fetchAll();
        
        sendResponse(true, 'Categories retrieved successfully', $categories, 200);
        
    } catch (PDOException $e) {
        logError("Get categories error: " . $e->getMessage());
        sendResponse(false, 'Failed to retrieve categories', null, 500);
    }
}
