<?php
/**
 * Profile API for Toko Baju Junata
 * Handles user profile and store information
 */

require_once 'config.php';

$requestData = getRequestData();
$action = isset($requestData['action']) ? $requestData['action'] : '';

switch ($action) {
    case 'get':
        getUserProfile($pdo, $requestData);
        break;
    
    case 'update':
        updateUserProfile($pdo, $requestData);
        break;
    
    case 'updatePassword':
        updatePassword($pdo, $requestData);
        break;
    
    case 'storeInfo':
        getStoreInfo($pdo);
        break;
    
    default:
        sendResponse(false, 'Invalid action', null, 400);
        break;
}

/**
 * Get user profile
 */
function getUserProfile($pdo, $data) {
    if (!isset($data['user_id']) || empty($data['user_id'])) {
        sendResponse(false, 'User ID is required', null, 400);
    }
    
    $user_id = intval($data['user_id']);
    
    try {
        $stmt = $pdo->prepare("
            SELECT id, nama, email, no_telp, alamat, foto_profil, created_at 
            FROM users 
            WHERE id = ?
        ");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        
        if (!$user) {
            sendResponse(false, 'User not found', null, 404);
        }
        
        // Get additional stats
        $stmt = $pdo->prepare("SELECT COUNT(*) as total_orders FROM orders WHERE user_id = ?");
        $stmt->execute([$user_id]);
        $stats = $stmt->fetch();
        
        $user['total_orders'] = $stats['total_orders'];
        
        sendResponse(true, 'Profile retrieved successfully', $user, 200);
        
    } catch (PDOException $e) {
        logError("Get user profile error: " . $e->getMessage());
        sendResponse(false, 'Failed to retrieve profile', null, 500);
    }
}

/**
 * Update user profile
 */
function updateUserProfile($pdo, $data) {
    if (!isset($data['user_id']) || empty($data['user_id'])) {
        sendResponse(false, 'User ID is required', null, 400);
    }
    
    $user_id = intval($data['user_id']);
    
    try {
        // Check if user exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        if (!$stmt->fetch()) {
            sendResponse(false, 'User not found', null, 404);
        }
        
        // Build update query dynamically based on provided fields
        $updateFields = [];
        $updateValues = [];
        
        if (isset($data['nama']) && !empty(trim($data['nama']))) {
            $updateFields[] = "nama = ?";
            $updateValues[] = sanitizeInput($data['nama']);
        }
        
        if (isset($data['no_telp'])) {
            $updateFields[] = "no_telp = ?";
            $updateValues[] = sanitizeInput($data['no_telp']);
        }
        
        if (isset($data['alamat'])) {
            $updateFields[] = "alamat = ?";
            $updateValues[] = sanitizeInput($data['alamat']);
        }
        
        if (isset($data['foto_profil'])) {
            $updateFields[] = "foto_profil = ?";
            $updateValues[] = sanitizeInput($data['foto_profil']);
        }
        
        if (empty($updateFields)) {
            sendResponse(false, 'No fields to update', null, 400);
        }
        
        $updateValues[] = $user_id;
        
        $sql = "UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($updateValues);
        
        // Get updated user data
        $stmt = $pdo->prepare("
            SELECT id, nama, email, no_telp, alamat, foto_profil, created_at 
            FROM users 
            WHERE id = ?
        ");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        
        sendResponse(true, 'Profile updated successfully', $user, 200);
        
    } catch (PDOException $e) {
        logError("Update user profile error: " . $e->getMessage());
        sendResponse(false, 'Failed to update profile', null, 500);
    }
}

/**
 * Update user password
 */
function updatePassword($pdo, $data) {
    $required = ['user_id', 'current_password', 'new_password'];
    $missing = validateRequired($data, $required);
    
    if (!empty($missing)) {
        sendResponse(false, 'Missing required fields: ' . implode(', ', $missing), null, 400);
    }
    
    $user_id = intval($data['user_id']);
    $current_password = $data['current_password'];
    $new_password = $data['new_password'];
    
    // Validate new password length
    if (strlen($new_password) < 6) {
        sendResponse(false, 'New password must be at least 6 characters', null, 400);
    }
    
    try {
        // Get current password
        $stmt = $pdo->prepare("SELECT password FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        
        if (!$user) {
            sendResponse(false, 'User not found', null, 404);
        }
        
        // Verify current password
        if (!password_verify($current_password, $user['password'])) {
            sendResponse(false, 'Current password is incorrect', null, 401);
        }
        
        // Hash new password
        $hashedPassword = password_hash($new_password, PASSWORD_BCRYPT);
        
        // Update password
        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $stmt->execute([$hashedPassword, $user_id]);
        
        sendResponse(true, 'Password updated successfully', null, 200);
        
    } catch (PDOException $e) {
        logError("Update password error: " . $e->getMessage());
        sendResponse(false, 'Failed to update password', null, 500);
    }
}

/**
 * Get store information
 */
function getStoreInfo($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM store_info LIMIT 1");
        $storeInfo = $stmt->fetch();
        
        if (!$storeInfo) {
            sendResponse(false, 'Store information not found', null, 404);
        }
        
        sendResponse(true, 'Store information retrieved successfully', $storeInfo, 200);
        
    } catch (PDOException $e) {
        logError("Get store info error: " . $e->getMessage());
        sendResponse(false, 'Failed to retrieve store information', null, 500);
    }
}
